<?php
/*
Plugin Name: Chemtech Exhibitor Registration - Final v2
Description: Final plugin: Full-form Step1 -> Summary Step2 -> Confirm & Payment Step3. Shortcode [chemtech_exhibitor_form]
Version: 1.1
Email:sangamkmaurya93@gmail.com
Author: Sangam Kumar Maurya
*/

if (!defined('ABSPATH')) exit;

global $wpdb;
if (!isset($wpdb)) $wpdb = $GLOBALS['wpdb'];
define('CT_FINAL2_TABLE', $wpdb->prefix . 'ct_exhibitors_final2');

/* ---------------- Activation: create table & default stall sizes ---------------- */
function ct_final2_activate() {
    global $wpdb;
    $table = CT_FINAL2_TABLE;
    $charset_collate = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE IF NOT EXISTS {$table} (
        id BIGINT(20) NOT NULL AUTO_INCREMENT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        data LONGTEXT NOT NULL,
        paid TINYINT(1) DEFAULT 0,
        payment_ref VARCHAR(255) DEFAULT NULL,
        PRIMARY KEY (id)
    ) {$charset_collate};";
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);

    // Default stall sizes (only if not present)
    if ( false === get_option('ct_stall_sizes') ) {
        $default = array(
            array(
                'id' => 'std-6sqm',
                'name' => '6 SQ. MTRS. (Standard)',
                'price_inr' => '3500',
                'price_usd' => '45',
                'link_inr' => '',
                'link_usd' => ''
            )
        );
        update_option('ct_stall_sizes', $default);
    }
}
register_activation_hook(__FILE__, 'ct_final2_activate');

/* ---------------- Admin menu ---------------- */
add_action('admin_menu', 'ct_final2_admin_menu');
function ct_final2_admin_menu() {
    add_menu_page('Exhibitor Registrations', 'Exhibitor Registrations', 'manage_options', 'ct_final2_entries', 'ct_final2_admin_page', 'dashicons-list-view', 26);
    add_submenu_page('ct_final2_entries', 'Settings', 'Settings', 'manage_options', 'ct_final2_settings', 'ct_final2_settings_page');
}

/* ---------------- Settings page (now includes Stall Sizes management) ---------------- */
function ct_final2_settings_page() {
    if (!current_user_can('manage_options')) return;

    // Handle saving global default prices/links (legacy single-price still present)
    if (isset($_POST['ct_final2_save']) && check_admin_referer('ct_final2_save_nonce')) {
        update_option('ct_final2_price_inr', sanitize_text_field($_POST['ct_final2_price_inr']));
        update_option('ct_final2_price_usd', sanitize_text_field($_POST['ct_final2_price_usd']));
        update_option('ct_final2_link_inr', esc_url_raw($_POST['ct_final2_link_inr']));
        update_option('ct_final2_link_usd', esc_url_raw($_POST['ct_final2_link_usd']));
        echo '<div class="updated"><p>Saved.</p></div>';
    }

    // Handle add stall size
    if (isset($_POST['ct_stall_add']) && check_admin_referer('ct_stall_sizes_nonce')) {
        $sizes = get_option('ct_stall_sizes', array());
        $new = array(
            'id' => sanitize_text_field($_POST['stall_id']),
            'name' => sanitize_text_field($_POST['stall_name']),
            'price_inr' => sanitize_text_field($_POST['stall_price_inr']),
            'price_usd' => sanitize_text_field($_POST['stall_price_usd']),
            'link_inr' => esc_url_raw($_POST['stall_link_inr']),
            'link_usd' => esc_url_raw($_POST['stall_link_usd'])
        );
        // prevent duplicate id
        $exists = false;
        foreach ($sizes as $s) { if (isset($s['id']) && $s['id'] === $new['id']) { $exists = true; break; } }
        if ($exists) {
            echo '<div class="error"><p>Stall ID already exists. Use unique ID.</p></div>';
        } else {
            $sizes[] = $new;
            update_option('ct_stall_sizes', $sizes);
            echo '<div class="updated"><p>Stall size added.</p></div>';
        }
    }

    // Handle delete stall size
    if (isset($_POST['ct_stall_delete']) && check_admin_referer('ct_stall_sizes_nonce')) {
        $del_id = sanitize_text_field($_POST['delete_stall_id']);
        $sizes = get_option('ct_stall_sizes', array());
        $filtered = array();
        foreach ($sizes as $s) {
            if (!isset($s['id']) || $s['id'] === $del_id) continue;
            $filtered[] = $s;
        }
        update_option('ct_stall_sizes', $filtered);
        echo '<div class="updated"><p>Stall removed.</p></div>';
    }

    $inr = get_option('ct_final2_price_inr','3500');
    $usd = get_option('ct_final2_price_usd','45');
    $link_inr = get_option('ct_final2_link_inr','');
    $link_usd = get_option('ct_final2_link_usd','');
    $sizes = get_option('ct_stall_sizes', array());
    ?>
    <div class="wrap"><h1>Chemtech Exhibitor - Settings</h1>

    <h2>Legacy Default Price / Links</h2>
    <form method="post">
      <?php wp_nonce_field('ct_final2_save_nonce'); ?>
      <table class="form-table">
        <tr><th>Price (INR)</th><td><input name="ct_final2_price_inr" value="<?php echo esc_attr($inr); ?>" /></td></tr>
        <tr><th>Price (USD)</th><td><input name="ct_final2_price_usd" value="<?php echo esc_attr($usd); ?>" /></td></tr>
        <tr><th>INR Payout Link</th><td><input style="width:100%" name="ct_final2_link_inr" value="<?php echo esc_attr($link_inr); ?>" /></td></tr>
        <tr><th>USD Payout Link</th><td><input style="width:100%" name="ct_final2_link_usd" value="<?php echo esc_attr($link_usd); ?>" /></td></tr>
      </table>
      <p class="submit"><button class="button button-primary" type="submit" name="ct_final2_save">Save Settings</button></p>
    </form>

    <hr/>

    <h2>Stall Sizes (Manage)</h2>
    <p>Add multiple stall sizes. These will populate the frontend dropdown and control price & payment links per size.</p>

    <form method="post" style="max-width:800px;">
      <?php wp_nonce_field('ct_stall_sizes_nonce'); ?>
      <table class="form-table">
        <tr><th>Unique ID (slug)</th><td><input name="stall_id" required placeholder="e.g. std-6sqm" /></td></tr>
        <tr><th>Display name</th><td><input name="stall_name" required placeholder="6 SQ. MTRS. (Standard)" /></td></tr>
        <tr><th>Price (INR)</th><td><input name="stall_price_inr" placeholder="e.g. 3500" /></td></tr>
        <tr><th>Price (USD)</th><td><input name="stall_price_usd" placeholder="e.g. 45" /></td></tr>
        <tr><th>Payment link (INR)</th><td><input name="stall_link_inr" style="width:100%" placeholder="https://..." /></td></tr>
        <tr><th>Payment link (USD)</th><td><input name="stall_link_usd" style="width:100%" placeholder="https://..." /></td></tr>
      </table>
      <p><button class="button button-primary" name="ct_stall_add" type="submit">Add Stall Size</button></p>
    </form>

    <h3>Existing Stall Sizes</h3>
    <table class="widefat" style="max-width:900px;">
      <thead><tr><th>ID</th><th>Name</th><th>INR</th><th>USD</th><th>Links</th><th>Action</th></tr></thead>
      <tbody>
        <?php foreach ($sizes as $s): ?>
          <tr>
            <td><?php echo esc_html($s['id']); ?></td>
            <td><?php echo esc_html($s['name']); ?></td>
            <td><?php echo esc_html($s['price_inr']); ?></td>
            <td><?php echo esc_html($s['price_usd']); ?></td>
            <td>
              IN: <?php echo esc_html($s['link_inr']); ?><br/>
              USD: <?php echo esc_html($s['link_usd']); ?>
            </td>
            <td>
              <form method="post" onsubmit="return confirm('Delete this stall size?');">
                <?php wp_nonce_field('ct_stall_sizes_nonce'); ?>
                <input type="hidden" name="delete_stall_id" value="<?php echo esc_attr($s['id']); ?>" />
                <button class="button button-small" name="ct_stall_delete" type="submit">Delete</button>
              </form>
            </td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>

    </div>
    <?php
}

/* ===== Hidden columns list (unchanged) ===== */
if (!defined('ABSPATH')) { /* silence */ }
if (!isset($ct_final2_hidden_cols)) {
    $ct_final2_hidden_cols = array('paid','payment_ref');
}
if (!function_exists('ct_final2_parse_data_field')) {
    function ct_final2_parse_data_field($raw) {
        if (is_array($raw)) return $raw;
        $d = json_decode($raw, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($d)) return $d;
        $un = @unserialize($raw);
        if ($un !== false && is_array($un)) return $un;
        return array('_raw_data' => (string)$raw);
    }
}

/* ---------------- Bulk delete & export handlers (unchanged) ---------------- */
/* (keeping existing code - not duplicated here for brevity; original handlers remain intact) */

/* ---------------- Admin page + table (unchanged except export/header) ---------------- */

function ct_final2_admin_page() {
    if (!current_user_can('manage_options')) return;
    global $wpdb, $ct_final2_hidden_cols;
    $table = defined('CT_FINAL2_TABLE') ? CT_FINAL2_TABLE : $wpdb->prefix.'ct_final2';

    // Parameters for search, filter, sort, pagination
    $search = isset($_GET['ct_search']) ? sanitize_text_field($_GET['ct_search']) : '';
    $country = isset($_GET['ct_country']) ? sanitize_text_field($_GET['ct_country']) : '';
    $city = isset($_GET['ct_city']) ? sanitize_text_field($_GET['ct_city']) : '';
    $date_from = isset($_GET['ct_date_from']) ? sanitize_text_field($_GET['ct_date_from']) : '';
    $date_to = isset($_GET['ct_date_to']) ? sanitize_text_field($_GET['ct_date_to']) : '';
    $sort = isset($_GET['ct_sort']) ? sanitize_text_field($_GET['ct_sort']) : 'id';
    $order = isset($_GET['ct_order']) && in_array(strtoupper($_GET['ct_order']), array('ASC','DESC')) ? strtoupper($_GET['ct_order']) : 'DESC';
    $paged = isset($_GET['paged']) ? max(1,intval($_GET['paged'])) : 1;
    $per_page = 20;
    $offset = ($paged - 1) * $per_page;

    // Build WHERE clauses based on filters (data is JSON in data column)
    $where = array();
    $params = array();

    if ($search !== '') {
        $s = '%'.$wpdb->esc_like($search).'%';
        $where[] = " (data LIKE %s OR data LIKE %s OR data LIKE %s OR data LIKE %s) ";
        $params[] = $s; $params[] = $s; $params[] = $s; $params[] = $s;
    }
    if ($country !== '') {
        $where[] = " (data LIKE %s) ";
        $params[] = '%"country":"'.$country.'"%';
    }
    if ($city !== '') {
        $where[] = " (data LIKE %s) ";
        $params[] = '%"city":"'.$city.'"%';
    }
    if ($date_from !== '') {
        $where[] = " (created_at >= %s) ";
        $params[] = $date_from.' 00:00:00';
    }
    if ($date_to !== '') {
        $where[] = " (created_at <= %s) ";
        $params[] = $date_to.' 23:59:59';
    }

    $where_sql = '';
    if (!empty($where)) {
        $where_sql = 'WHERE ' . implode(' AND ', $where);
    }

    $allowed_sorts = array('id'=>'id','created_at'=>'created_at');
    $order_by = isset($allowed_sorts[$sort]) ? $allowed_sorts[$sort] : 'id';

    // Total count for pagination
    if (!empty($params)) {
        $count_sql = $wpdb->prepare("SELECT COUNT(*) FROM {$table} {$where_sql}", $params);
        $total = $wpdb->get_var($count_sql);
    } else {
        $count_sql = "SELECT COUNT(*) FROM {$table} {$where_sql}";
        $total = $wpdb->get_var($count_sql);
    }

    // Fetch rows with limit
    $sql = "SELECT * FROM {$table} {$where_sql} ORDER BY {$order_by} {$order} LIMIT %d OFFSET %d";
    $params_with_limit = $params;
    $params_with_limit[] = $per_page;
    $params_with_limit[] = $offset;
    if (!empty($params)) {
        $rows = $wpdb->get_results($wpdb->prepare($sql, $params_with_limit));
    } else {
        $rows = $wpdb->get_results($wpdb->prepare($sql, array($per_page, $offset)));
    }

    // Fetch distinct countries and cities for filters (from JSON - simple approach)
    $all_rows = $wpdb->get_results("SELECT data FROM {$table} LIMIT 1000");
    $countries = array(); $cities = array();
    foreach($all_rows as $ar) {
        $d = json_decode($ar->data, true);
        if (is_array($d)) {
            if (!empty($d['country'])) $countries[] = $d['country'];
            if (!empty($d['city'])) $cities[] = $d['city'];
        }
    }
    $countries = array_unique($countries); sort($countries);
    $cities = array_unique($cities); sort($cities);

    echo '<div class="wrap"><h1>Exhibitor Entries</h1>';

    // Notice messages
    if (isset($_GET['deleted'])) {
        printf('<div class="notice notice-success is-dismissible"><p>%d entries deleted.</p></div>', intval($_GET['deleted']));
    }

    // Filters form and export buttons
    $current_url = esc_url(remove_query_arg(array('paged','ct_search','ct_country','ct_city','ct_date_from','ct_date_to','ct_sort','ct_order')));
    echo '<form method="get" class="ct-filter-form" style="margin-bottom:15px;">';
    echo '<input type="hidden" name="page" value="ct_final2_entries" />';
    echo 'Search: <input type="search" name="ct_search" value="'.esc_attr($search).'" /> ';
    echo 'Country: <select name="ct_country"><option value="">All</option>';
    foreach($countries as $c) echo '<option value="'.esc_attr($c).'" '.selected($country,$c,false).'>'.esc_html($c).'</option>';
    echo '</select> ';
    echo 'City: <select name="ct_city"><option value="">All</option>';
    foreach($cities as $c2) echo '<option value="'.esc_attr($c2).'" '.selected($city,$c2,false).'>'.esc_html($c2).'</option>';
    echo '</select> ';
    echo 'From: <input type="date" name="ct_date_from" value="'.esc_attr($date_from).'" /> ';
    echo 'To: <input type="date" name="ct_date_to" value="'.esc_attr($date_to).'" /> ';
    echo ' <input type="submit" class="button" value="Filter" /> ';
    echo '</form>';

    // Export buttons (All and Selected)
    echo '<p style="margin-bottom:12px;">';
    echo '<a class="button" href="'.esc_url(add_query_arg('ct_final2_export','csv')).'">Export All (CSV)</a> ';
    // Selected export form button (uses admin-post.php)
    echo '<form id="ct_export_selected_form" method="post" action="'.esc_url(admin_url('admin-post.php')).'" style="display:inline-block;margin-left:10px;">';
    wp_nonce_field('ct_final2_export_selected_action','ct_final2_export_selected_nonce');
    echo '<input type="hidden" name="action" value="ct_final2_export_selected" />';
    echo '<input type="submit" class="button button-primary" value="Export Selected (CSV)" onclick="return submitSelectedExport();" />';
    echo '</form>';
    echo '</p>';

    // Bulk delete form + table
    echo '<form id="ct_bulk_form" method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
    echo '<input type="hidden" name="action" value="ct_final2_bulk_delete" />';
    wp_nonce_field('ct_final2_bulk_action','ct_final2_bulk_nonce');
    echo '<p><input type="submit" class="button button-danger" value="Delete Selected" onclick="return confirm(\'Are you sure you want to delete selected entries?\');" /></p>';

    // Table headers with sorting links
    $base_q = array('page'=>'ct_final2_entries','ct_search'=>$search,'ct_country'=>$country,'ct_city'=>$city,'ct_date_from'=>$date_from,'ct_date_to'=>$date_to);
    echo '<table class="wp-list-table widefat fixed striped"><thead><tr>';
    echo '<th style="width:30px;"><input type="checkbox" id="ct_final2_select_all" /></th>';
    $cols = array('ID'=>'id','Created At'=>'created_at','Data'=>'data');
    foreach($cols as $label=>$colkey) {
        $dir = ($sort==$colkey && $order=='ASC') ? 'DESC' : 'ASC';
        $link = add_query_arg(array_merge($base_q,array('ct_sort'=>$colkey,'ct_order'=>$dir)));
        echo '<th><a href="'.esc_url($link).'">'.esc_html($label).'</a></th>';
    }
    echo '</tr></thead><tbody>';

    foreach ($rows as $r) {
        $d = ct_final2_parse_data_field($r->data);
        if (is_array($d)) {
            foreach ($ct_final2_hidden_cols as $hk) {
                if (isset($d[$hk])) unset($d[$hk]);
            }
        }
        echo '<tr>';
        echo '<td><input type="checkbox" name="ct_ids[]" value="'.intval($r->id).'" class="ct_row_checkbox" /></td>';
        echo '<td>'.esc_html($r->id).'</td>';
        echo '<td>'.esc_html($r->created_at).'</td>';
        echo '<td style="max-width:600px;"><pre style="white-space:pre-wrap;word-wrap:break-word;margin:0;padding:0;">';
        if (is_array($d)) {
            foreach ($d as $k=>$v) {
                if (is_array($v) || is_object($v)) $v = json_encode($v);
                echo esc_html($k) . ': ' . esc_html($v) . "\n";
            }
        } else {
            echo esc_html((string)$d);
        }
        echo '</pre></td>';
        echo '</tr>';
    }

    echo '</tbody></table>';
    echo '</form>';

    // Pagination links
    $total_pages = ceil($total / $per_page);
    if ($total_pages > 1) {
        echo '<div class="tablenav"><div class="tablenav-pages">';
        for ($p=1;$p<=$total_pages;$p++){
            $link = add_query_arg(array_merge($base_q,array('paged'=>$p)));
            if ($p==$paged) echo '<span class="page-numbers current">'.intval($p).'</span> ';
            else echo '<a class="page-numbers" href="'.esc_url($link).'">'.intval($p).'</a> ';
        }
        echo '</div></div>';
    }

    // JS select all + export selected submission
    echo '<script>
    document.addEventListener("DOMContentLoaded", function(){
        var sel = document.getElementById("ct_final2_select_all");
        if (!sel) return;
        sel.addEventListener("change", function(){
            var checked = this.checked;
            document.querySelectorAll(".ct_row_checkbox").forEach(function(cb){ cb.checked = checked; });
        });
    });

    function submitSelectedExport(){
        var exportForm = document.getElementById("ct_export_selected_form");
        // Remove any previously appended hidden inputs
        var existing = exportForm.querySelectorAll("input[name=\'ct_ids[]\']");
        existing.forEach(function(e){ e.parentNode.removeChild(e); });
        var checked = document.querySelectorAll(".ct_row_checkbox:checked");
        if (checked.length === 0) {
            alert("Please select at least one row to export.");
            return false;
        }
        checked.forEach(function(cb){
            var h = document.createElement("input");
            h.type = "hidden";
            h.name = "ct_ids[]";
            h.value = cb.value;
            exportForm.appendChild(h);
        });
        return true; // allow submit
    }
    </script>';

    echo '</div>';
}

/* ---------------- Shortcode and assets ---------------- */
add_shortcode('chemtech_exhibitor_form', 'ct_final2_render_form');
function ct_final2_render_form($atts) {
    wp_enqueue_script('jquery');
    wp_enqueue_script('ct_final2_js', plugins_url('assets/ct_final2.js', __FILE__), array('jquery'), null, true);

    // localize: include existing legacy prices + dynamic stall sizes map
    $stall_sizes = get_option('ct_stall_sizes', array());
    // make map keyed by id for easier JS lookup
    $stall_map = array();
    foreach ($stall_sizes as $s) {
        if (isset($s['id'])) $stall_map[$s['id']] = $s;
    }

    wp_localize_script('ct_final2_js', 'ctFinal2', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('ct_final2_nonce'),
        'price_inr' => get_option('ct_final2_price_inr','3500'),
        'price_usd' => get_option('ct_final2_price_usd','45'),
        'link_inr' => get_option('ct_final2_link_inr',''),
        'link_usd' => get_option('ct_final2_link_usd',''),
        'stall_sizes' => $stall_map
    ));
    wp_enqueue_style('ct_final2_css', plugins_url('assets/ct_final2.css', __FILE__));
    ob_start();
    include plugin_dir_path(__FILE__).'templates/form-template.php';
    return ob_get_clean();
}

/* ---------------- AJAX save handler ---------------- */
add_action('wp_ajax_nopriv_ct_final2_save', 'ct_final2_save');
add_action('wp_ajax_ct_final2_save', 'ct_final2_save');
function ct_final2_save() {
    if ( empty($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ct_final2_nonce') ) {
        wp_send_json_error(array('message'=>'Invalid request'), 400);
    }

    // expected fields (including new stall-related ones)
    $expected = array(
        'salutation','first_name','last_name','designation','company','address1','address2',
        'city','pincode','country','specify','mobile','landline','email','website','area',
        'stall_size','stall_price_inr','stall_price_usd','stall_link_inr','stall_link_usd'
    );
    $data = array();
    foreach ($expected as $k) {
        $data[$k] = isset($_POST[$k]) ? sanitize_text_field($_POST[$k]) : '';
    }

    $required = array('salutation','first_name','last_name','company','address1','city','pincode','country','mobile','email','website','stall_size');
    foreach ($required as $r) {
        if ( empty($data[$r]) ) {
            wp_send_json_error(array('message'=>"Field {$r} is required"), 422);
        }
    }
    if (!is_email($data['email'])) {
        wp_send_json_error(array('message'=>'Invalid email'), 422);
    }

    // augment stall display name from saved sizes (if found)
    $stall_display = $data['stall_size'];
    $stall_sizes = get_option('ct_stall_sizes', array());
    foreach ($stall_sizes as $s) {
        if (isset($s['id']) && $s['id'] === $data['stall_size']) {
            $stall_display = isset($s['name']) ? $s['name'] : $s['id'];
            // if particular price or link not provided by front-end, fill it from setting
            if (empty($data['stall_price_inr'])) $data['stall_price_inr'] = isset($s['price_inr']) ? $s['price_inr'] : '';
            if (empty($data['stall_price_usd'])) $data['stall_price_usd'] = isset($s['price_usd']) ? $s['price_usd'] : '';
            if (empty($data['stall_link_inr'])) $data['stall_link_inr'] = isset($s['link_inr']) ? $s['link_inr'] : '';
            if (empty($data['stall_link_usd'])) $data['stall_link_usd'] = isset($s['link_usd']) ? $s['link_usd'] : '';
            break;
        }
    }
    
    /* --------------------------------------------------
   SELECT FINAL STALL PRICE & LINK BASED ON COUNTRY
-------------------------------------------------- */

$country = strtolower($data['country']);

if ($country === "india" || $country === "in") {
    $final_price = $data['stall_price_inr'];
    $final_link  = $data['stall_link_inr'];
} else {
    $final_price = $data['stall_price_usd'];
    $final_link  = $data['stall_link_usd'];
}

 // Only save selected values
$data['stall_price'] = $final_price;
$data['stall_link']  = $final_link;

// Remove unused fields so they do not get saved in DB
unset($data['stall_price_inr']);
unset($data['stall_price_usd']);
unset($data['stall_link_inr']);
unset($data['stall_link_usd']);


    // Save submission into DB
    global $wpdb;
    $table = CT_FINAL2_TABLE;
    $save_row = array(
        'data' => maybe_serialize($data),
        'paid' => 0,
        'payment_ref' => null
    );
    $formats = array('%s','%d','%s');
    $ok = $wpdb->insert($table, $save_row, array('%s','%d','%s'));
    if (!$ok) {
        wp_send_json_error(array('message'=>'DB save failed'), 500);
    }
    $id = $wpdb->insert_id;

    // Build email body including stall info and payment links
    $to = $data['email'];
    $subj = 'Scale 2026 - Registration Received';
    $body  = "========================================\n";
    $body .= "📝 REGISTRATION DETAILS\n";
    $body .= "========================================\n\n";
    $body .= "Salutation       : {$data['salutation']}\n";
    $body .= "First Name       : {$data['first_name']}\n";
    $body .= "Last Name        : {$data['last_name']}\n";
    $body .= "Designation      : {$data['designation']}\n";
    $body .= "Company Name     : {$data['company']}\n";
    $body .= "Address Line 1   : {$data['address1']}\n";
    $body .= "Address Line 2   : {$data['address2']}\n";
    $body .= "City             : {$data['city']}\n";
    $body .= "Pincode          : {$data['pincode']}\n";
    $body .= "Country          : {$data['country']}\n";
    $body .= "Specify (if any) : {$data['specify']}\n";
    $body .= "Mobile Number    : {$data['mobile']}\n";
    $body .= "Landline Number  : {$data['landline']}\n";
    $body .= "Email Address    : {$data['email']}\n";
    $body .= "Website          : {$data['website']}\n\n";

    // Stall info
$body .= "----------------------------------------\n";
$body .= "📦 SELECTED STALL\n";
$body .= "----------------------------------------\n\n";

$body .= "Stall Selected : {$stall_display}\n";

// Detect selected currency based on country
$country = strtolower($data['country']);
$currency = ($country === "india" || $country === "in") ? "INR" : "USD";
$price    = isset($data['stall_price']) ? $data['stall_price'] : '';
$link     = isset($data['stall_link'])  ? $data['stall_link']  : '';

// Email output – only selected price + link
$body .= "Price ({$currency}) : {$currency} {$price}\n";
//$body .= "Payment Link        : {$link}\n\n";


    $body .= "Registration ID: {$id}\n";
    $body .= "----------------------------------------\n\n";

    $body .= "⚠️ *Important Information:*\n";
    $body .= "Your registration form has been received successfully.\n";
    $body .= "However, your registration will be considered **confirmed only after successful payment**.\n\n";
    $body .= "💡 If you have already completed the payment, please ignore this message.\n";
    $body .= "Your registration will be treated as confirmed automatically after verification.\n\n";
    $body .= "========================================\n";
    $body .= "Thank you for your interest in Scale 2026!\n";
    $body .= "Team Scale 2026\n";
    $body .= "========================================\n";

    wp_mail($to, $subj, $body);
    wp_mail(get_option('admin_email'), 'Scale 2026 - New Registration #'.$id, $body);

    wp_send_json_success(array('id'=>$id));
}

/* ---------------- Export CSV routes preserved from original file ---------------
   Note: The original export/bulk-delete code remains unchanged and will export saved
   serialized data which now includes stall fields. (No changes required here.)
*/

/* End of file */
/* ---------------- FIX: Export Selected, Bulk Delete, Export All CSV handlers ---------------- */

/**
 * Bulk delete selected entries (admin-post handler)
 * Expects POST: action=ct_final2_bulk_delete, ct_ids[] array, nonce ct_final2_bulk_nonce
 */
add_action('admin_post_ct_final2_bulk_delete', function() {
    if (!current_user_can('manage_options')) wp_die('Unauthorized');
    if ( ! isset($_POST['ct_final2_bulk_nonce']) || ! wp_verify_nonce($_POST['ct_final2_bulk_nonce'], 'ct_final2_bulk_action') ) {
        wp_die('Security check failed');
    }
    if ( empty($_POST['ct_ids']) || ! is_array($_POST['ct_ids']) ) {
        wp_redirect(admin_url('admin.php?page=ct_final2_entries'));
        exit;
    }

    global $wpdb;
    $table = defined('CT_FINAL2_TABLE') ? CT_FINAL2_TABLE : $wpdb->prefix.'ct_final2';
    // sanitize ids as integers and build IN list
    $ids = array_map('intval', $_POST['ct_ids']);
    if (empty($ids)) {
        wp_redirect(admin_url('admin.php?page=ct_final2_entries'));
        exit;
    }
    $ids_list = implode(',', $ids);
    $wpdb->query("DELETE FROM {$table} WHERE id IN ({$ids_list})");

    wp_redirect(admin_url('admin.php?page=ct_final2_entries&deleted='.count($ids)));
    exit;
});

/**
 * Export selected rows to CSV (admin-post handler)
 * Expects POST: action=ct_final2_export_selected, ct_ids[] array, nonce ct_final2_export_selected_nonce
 */
add_action('admin_post_ct_final2_export_selected', function() {
    if (!current_user_can('manage_options')) wp_die('Unauthorized');
    if ( ! isset($_POST['ct_final2_export_selected_nonce']) || ! wp_verify_nonce($_POST['ct_final2_export_selected_nonce'], 'ct_final2_export_selected_action') ) {
        wp_die('Security check failed');
    }
    if ( empty($_POST['ct_ids']) || ! is_array($_POST['ct_ids']) ) {
        wp_redirect(admin_url('admin.php?page=ct_final2_entries'));
        exit;
    }

    global $wpdb;
    $table = defined('CT_FINAL2_TABLE') ? CT_FINAL2_TABLE : $wpdb->prefix.'ct_final2';
    $ids = array_map('intval', $_POST['ct_ids']);
    if (empty($ids)) {
        wp_redirect(admin_url('admin.php?page=ct_final2_entries'));
        exit;
    }
    $ids_list = implode(',', $ids);

    $rows = $wpdb->get_results("SELECT * FROM {$table} WHERE id IN ({$ids_list}) ORDER BY id DESC", ARRAY_A);
    if (empty($rows)) {
        wp_redirect(admin_url('admin.php?page=ct_final2_entries'));
        exit;
    }

    // Output CSV
    while (ob_get_level()) ob_end_clean();
    nocache_headers();
    $fname = 'exhibitors_selected_' . date('Ymd') . '.csv';
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="'.$fname.'"');
    header('Pragma: no-cache');
    header('Expires: 0');

    $out = fopen('php://output', 'w');
    // BOM
    fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));

    // Headers (include stall fields)
    $headers = array(
        'ID','Created At','salutation','first_name','last_name','designation','company',
        'address1','address2','city','pincode','country','specify','mobile','landline','email','website','area',
        // stall-related
        'stall_size','stall_price_inr','stall_price_usd','stall_link_inr','stall_link_usd'
    );
    fputcsv($out, $headers);

    foreach ($rows as $r) {
        $data = maybe_unserialize($r['data']);
        if (!is_array($data)) $data = array();

        // Build row values in the same order as headers
        $row_vals = array();
        $row_vals[] = $r['id'];
        $row_vals[] = $r['created_at'];

        $fields = array('salutation','first_name','last_name','designation','company','address1','address2','city','pincode','country','specify','mobile','landline','email','website','area');
        foreach ($fields as $f) {
            $row_vals[] = isset($data[$f]) ? $data[$f] : '';
        }

        // stall fields
        $row_vals[] = isset($data['stall_size']) ? $data['stall_size'] : '';
        $row_vals[] = isset($data['stall_price_inr']) ? $data['stall_price_inr'] : '';
        $row_vals[] = isset($data['stall_price_usd']) ? $data['stall_price_usd'] : '';
        $row_vals[] = isset($data['stall_link_inr']) ? $data['stall_link_inr'] : '';
        $row_vals[] = isset($data['stall_link_usd']) ? $data['stall_link_usd'] : '';

        fputcsv($out, $row_vals);
    }
    fclose($out);
    exit;
});

/**
 * Export All CSV via GET parameter (admin_init)
 * Usage: admin page link with ?ct_final2_export=csv
 */
add_action('admin_init', function() {
    if (!current_user_can('manage_options')) return;
    if (isset($_GET['ct_final2_export']) && $_GET['ct_final2_export'] === 'csv') {
        global $wpdb;
        $table = defined('CT_FINAL2_TABLE') ? CT_FINAL2_TABLE : $wpdb->prefix.'ct_final2';
        $rows = $wpdb->get_results("SELECT * FROM {$table} ORDER BY id DESC", ARRAY_A);
        if (empty($rows)) {
            wp_redirect(admin_url('admin.php?page=ct_final2_entries'));
            exit;
        }

        // Output CSV
        while (ob_get_level()) ob_end_clean();
        nocache_headers();
        $fname = 'exhibitors_all_' . date('Ymd') . '.csv';
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="'.$fname.'"');
        header('Pragma: no-cache');
        header('Expires: 0');

        $out = fopen('php://output', 'w');
        fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));

        $headers = array(
            'ID','Created At','salutation','first_name','last_name','designation','company',
            'address1','address2','city','pincode','country','specify','mobile','landline','email','website','area',
            // stall-related
            'stall_size','stall_price_inr','stall_price_usd','stall_link_inr','stall_link_usd'
        );
        fputcsv($out, $headers);

        foreach ($rows as $r) {
            $data = maybe_unserialize($r['data']);
            if (!is_array($data)) $data = array();

            $row_vals = array();
            $row_vals[] = $r['id'];
            $row_vals[] = $r['created_at'];

            $fields = array('salutation','first_name','last_name','designation','company','address1','address2','city','pincode','country','specify','mobile','landline','email','website','area');
            foreach ($fields as $f) {
                $row_vals[] = isset($data[$f]) ? $data[$f] : '';
            }

            $row_vals[] = isset($data['stall_size']) ? $data['stall_size'] : '';
            $row_vals[] = isset($data['stall_price_inr']) ? $data['stall_price_inr'] : '';
            $row_vals[] = isset($data['stall_price_usd']) ? $data['stall_price_usd'] : '';
            $row_vals[] = isset($data['stall_link_inr']) ? $data['stall_link_inr'] : '';
            $row_vals[] = isset($data['stall_link_usd']) ? $data['stall_link_usd'] : '';

            fputcsv($out, $row_vals);
        }
        fclose($out);
        exit;
    }
  });

