<?php
/*
Plugin Name: Team Grid Plugin (with Font Awesome)
Description: Display team members in a responsive grid with popups and backend management (includes Font Awesome icons). Shortcode: [team_grid]
Version: 1.2
Author: Sangam Kumar Maurya
*/

if (!defined('ABSPATH')) exit;

/* ---------------- Register Custom Post Type ---------------- */
function tgp_register_team_post_type() {
    $labels = array(
        'name' => 'Team Members',
        'singular_name' => 'Team Member',
        'add_new' => 'Add New Member',
        'all_items' => 'All Team Members',
        'add_new_item' => 'Add New Member',
        'edit_item' => 'Edit Member',
        'new_item' => 'New Member',
        'view_item' => 'View Member',
        'search_items' => 'Search Members',
        'not_found' => 'No members found',
    );

    $args = array(
        'labels' => $labels,
        'public' => false,
        'show_ui' => true,
        'show_in_menu' => true,
        'menu_icon' => 'dashicons-groups',
        'supports' => array('title', 'thumbnail'),
    );

    register_post_type('team_member', $args);
}
add_action('init', 'tgp_register_team_post_type');

/* ---------------- Add Meta Boxes ---------------- */
function tgp_add_team_meta_boxes() {
    add_meta_box('tgp_team_details', 'Team Details', 'tgp_team_details_callback', 'team_member', 'normal', 'default');
}
add_action('add_meta_boxes', 'tgp_add_team_meta_boxes');

function tgp_team_details_callback($post) {
    wp_nonce_field( 'tgp_save_meta', 'tgp_meta_nonce' );
    $designation = get_post_meta($post->ID, '_designation', true);
    $description = get_post_meta($post->ID, '_description', true);
    $linkedin = get_post_meta($post->ID, '_linkedin', true);
    $twitter = get_post_meta($post->ID, '_twitter', true);
    ?>
    <p><label><strong>Designation:</strong></label><br/>
    <input type="text" name="designation" value="<?php echo esc_attr($designation); ?>" class="widefat"></p>

    <p><label><strong>Description:</strong></label><br/>
    <textarea name="description" rows="4" class="widefat"><?php echo esc_textarea($description); ?></textarea></p>

    <p><label><strong>LinkedIn URL:</strong></label><br/>
    <input type="url" name="linkedin" value="<?php echo esc_url($linkedin); ?>" class="widefat"></p>

    <p><label><strong>Twitter URL:</strong></label><br/>
    <input type="url" name="twitter" value="<?php echo esc_url($twitter); ?>" class="widefat"></p>
    <?php
}

function tgp_save_team_meta($post_id) {
    if ( ! isset( $_POST['tgp_meta_nonce'] ) || ! wp_verify_nonce( $_POST['tgp_meta_nonce'], 'tgp_save_meta' ) ) {
        return;
    }
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;

    $fields = ['designation','description','linkedin','twitter'];
    foreach($fields as $field){
        if(array_key_exists($field,$_POST)){
            update_post_meta($post_id, '_'.$field, sanitize_text_field($_POST[$field]));
        }
    }
}
add_action('save_post', 'tgp_save_team_meta');

/* ---------------- Enqueue CSS, JS & Font Awesome ---------------- */
function tgp_enqueue_assets() {
    wp_enqueue_style('font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css');
    wp_enqueue_style('tgp-style', plugin_dir_url(__FILE__) . 'style.css', array(), '1.2');
    wp_enqueue_script('tgp-script', plugin_dir_url(__FILE__) . 'script.js', array('jquery'), '1.2', true);
}
add_action('wp_enqueue_scripts', 'tgp_enqueue_assets');

/* ---------------- Shortcode ---------------- */
function tgp_team_shortcode() {
    $args = array('post_type' => 'team_member', 'posts_per_page' => -1);
    $query = new WP_Query($args);

    ob_start();
    echo '<div class="tgp-grid">';
    while ($query->have_posts()) : $query->the_post();
        $id = get_the_ID();
        $title = get_the_title();
        $designation = get_post_meta($id, '_designation', true);
        $description = get_post_meta($id, '_description', true);
        $linkedin = get_post_meta($id, '_linkedin', true);
        $twitter = get_post_meta($id, '_twitter', true);
        $img = get_the_post_thumbnail_url($id, 'medium');
        ?>
        <div class="tgp-card">
            <div class="tgp-imgbox">
                <img src="<?php echo esc_url($img); ?>" alt="<?php echo esc_attr($title); ?>">
            </div>
            <h3 class="tgp-name"><?php echo esc_html($title); ?></h3>
            <p class="tgp-designation"><?php echo esc_html($designation); ?></p>
            <button class="tgp-details-btn" data-id="<?php echo esc_attr($id); ?>">Know More</button>
        </div>

        <div class="tgp-popup" id="popup-<?php echo esc_attr($id); ?>">
            <div class="tgp-popup-content">
                <span class="tgp-close">&times;</span>
                <h2 class="popup-team-name"><?php echo esc_html($title); ?></h2>
                <h4 class="popup-team-desination"><?php echo esc_html($designation); ?></h4>
                <p><?php echo esc_html($description); ?></p>
                <div class="tgp-social">
                    <?php if($linkedin): ?><a href="<?php echo esc_url($linkedin); ?>" target="_blank"><i class="fab fa-linkedin"></i></a><?php endif; ?>
                    <?php if($twitter): ?><a href="<?php echo esc_url($twitter); ?>" target="_blank"><i class="fab fa-twitter"></i></a><?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    endwhile;
    echo '</div>';
    wp_reset_postdata();
    return ob_get_clean();
}
add_shortcode('team_grid', 'tgp_team_shortcode');
