/**
 * Plugin Name: Custom WooCommerce PDF Cart
 * Description: Custom WooCommerce cart with quantity, notifications, PDF quote generation, phone field, thumbnails, and thank-you screen.
 * Version: 4.5
 * Author: Sangam Kumar Maurya
 * Author Email: sangamkmaurya93@gmail.com
 */

jQuery(document).ready(function ($) {

    // ================= Mini-Cart Popup =================
    // Open / Close popup when cart button is clicked
    $(document).on('click', '#custom-cart-button', function () {
        $('#custom-cart-popup').toggleClass('open'); // Toggle 'open' class to show/hide popup
    });

    // Close popup if user clicks outside of popup
    $(document).on('click', function (e) {
        if (!$(e.target).closest('#custom-cart-popup, #custom-cart-button').length) {
            $('#custom-cart-popup').removeClass('open');
        }
    });


    // ================= Add Product to Custom Cart =================
    $(document).on('click', '.add-to-custom-cart', function (e) {
        e.preventDefault();

        const $btn = $(this);
        const productId = $btn.data('product-id');

        // Get quantity from input (default = 1 if not found)
        const qty = parseInt($btn.siblings('.custom-cart-qty').val()) || 1;

        // AJAX request to add product to custom cart
        $.post(customCartAjax.ajaxurl, {
            action: 'add_to_custom_cart',
            nonce: customCartAjax.nonce,
            product_id: productId,
            quantity: qty
        }, function (res) {
            if (res.success) {
                refreshUI(res.data);      // Update mini-cart & cart page UI
                notify(res.data.message); // Show notification message

                // Change button to "added" state immediately
                $btn.addClass('added').html('<i class="fa fa-check" aria-hidden="true"></i>');
            }
        });
    });


    // ================= Quantity Buttons (+ / -) =================
    $(document).on('click', '.qty-plus, .qty-minus', function () {
        const $li = $(this).closest('li');
        const id = $li.data('id');
        const $input = $li.find('.qty-input');
        const maxQty = parseInt($li.data('max-qty')) || 9999;
        let qty = parseInt($input.val()) || 1;

        if ($(this).hasClass('qty-plus')) {
            // Increment quantity (do not exceed max stock)
            if (qty >= maxQty) {
                notify('❌ Maximum quantity reached');
                return;
            }
            qty++;
        } else {
            // Decrement quantity (minimum = 1)
            if (qty <= 1) {
                notify('❌ Quantity cannot be less than 1');
                return;
            }
            qty--;
        }

        // Update input field and send AJAX update
        $input.val(qty);
        updateQtyAjax(id, qty, $li);
    });


    // ---------------- Quantity Input Validation for Mini-Cart & Product Pages ----------------
$(document).on('input', '.qty-input, .custom-cart-qty', function () {
    const $input = $(this);
    const $li = $input.closest('li');
    
    // Max stock (from data attribute if mini-cart, else default 9999)
    const maxQty = parseInt($li.data('max-qty')) || parseInt($input.attr('max')) || 9999; 
    const minQty = parseInt($input.attr('min')) || 1; // Minimum 1

    let val = parseInt($input.val());

    // Enforce minimum
    if (isNaN(val) || val < minQty) {
        val = minQty;
        notify(`❌ Minimum quantity is ${minQty}`);
    }

    // Enforce maximum
    if (val > maxQty) {
        val = maxQty;
        notify(`❌ Maximum quantity is ${maxQty}`);
    }

    $input.val(val);

    // Optional: AJAX update for mini-cart items
    if ($input.hasClass('qty-input') && $li.length) {
        clearTimeout($input.data('debounceTimer'));
        const timer = setTimeout(function () {
            validateAndUpdateQty($input); // Existing function to update via AJAX
        }, 500);
        $input.data('debounceTimer', timer);
    }
 });


    // Validate quantity immediately on blur/change
    $(document).on('change', '.qty-input', function () {
        const $input = $(this);
        clearTimeout($input.data('debounceTimer'));
        validateAndUpdateQty($input);
    });


    /**
     * Validate quantity input and update cart via AJAX
     * @param $input jQuery object of the quantity input field
     */
    function validateAndUpdateQty($input) {
        const $li = $input.closest('li');
        const id = $li.data('id');
        const maxQty = parseInt($li.data('max-qty')) || 9999;

        let qty = parseInt($input.val());

        // If blank, NaN, or <1 → set to 1
        if (isNaN(qty) || qty < 1) qty = 1;

        // Limit quantity to stock
        if (qty > maxQty) {
            qty = maxQty;
            notify(`❌ Only ${maxQty} items available in stock`);
        }

        $input.val(qty); // Normalize input
        updateQtyAjax(id, qty, $li); // Send AJAX update
    }


    // ================= Remove Item from Cart =================
    $(document).on('click', '.remove-item, .remove-item-page', function () {
        const id = $(this).closest('li').data('id');

        $.post(customCartAjax.ajaxurl, {
            action: 'remove_from_custom_cart',
            nonce: customCartAjax.nonce,
            product_id: id
        }, function (res) {
            if (res.success) {
                refreshUI(res.data);      // Update mini-cart & cart page UI
                notify(res.data.message); // Show notification
            }
        });
    });


    // ================= Proceed to Form Page =================
    $(document).on('click', '#proceed-to-form', function () {
        window.location.href = customCartAjax.form_page_url;
    });


    // ================= Update Quantity via AJAX =================
    function updateQtyAjax(productId, quantity, $li) {
        // Optional: add mini loader to $li here if needed

        $.post(customCartAjax.ajaxurl, {
            action: 'update_cart_quantity',
            nonce: customCartAjax.nonce,
            product_id: productId,
            quantity: quantity
        }, function (res) {
            if (res.success) {
                refreshUI(res.data);      // Update UI
                notify(res.data.message); // Show success message
            } else if (res.data && res.data.message) {
                // Server-side validation feedback
                notify(res.data.message);
            }
        });
    }


    // ================= Refresh UI: Mini-Cart & Cart Page =================
    function refreshUI(data) {
        // --- Update cart counters ---
        $('#cart-count').length && $('#cart-count').text(data.count);
        $('#cart-count-page').length && $('#cart-count-page').text(data.count);
        $('#cart-total').length && $('#cart-total').text(data.total);
        $('#cart-total-page').length && $('#cart-total-page').text(data.total);

        // --- Mini-Cart Popup HTML ---
        let popupHTML = '';
        data.items.forEach(item => {
            const maxAttr = item.stock || 9999;
            popupHTML += `
                <li data-id="${item.id}" data-max-qty="${maxAttr}">
                    <img src="${item.img || ''}" class="cart-thumb"/>
                    <div class="cart-info">
                        <strong>${item.title}</strong><br>
                        Qty:
                        <button class="qty-minus">➖</button>
                        <input type="number" class="qty-input" value="${item.qty}" min="1" max="${maxAttr}">
                        <button class="qty-plus">➕</button><br>
                        <span class="cart-price">${item.price}</span>
                    </div>
                    <button class="remove-item"><i class="fa fa-trash"></i></button>
                    </li>`;
        });
        $('.cart-items').html(popupHTML);

        // --- Cart Page HTML ---
        if ($('#cart-items-page').length) {
            let pageHTML = '';
            data.items.forEach(item => {
                const maxAttr = item.stock || 9999;
                pageHTML += `
                    <li data-id="${item.id}" data-max-qty="${maxAttr}">
                        <div class="cart-item-box">
                            <img src="${item.img || ''}" class="cart-item-thumb" alt="">
                            <div class="cart-item-content">
                                <div class="cart-item-title">${item.title}</div>
                                <div class="cart-item-details">
                                    Qty:
                                    <button class="qty-minus">➖</button>
                                    <input type="number" class="qty-input" value="${item.qty}" min="1" max="${maxAttr}">
                                    <button class="qty-plus">➕</button>
                                    <span class="cart-price">${item.price}</span>
                                </div>
                                <div class="cart-item-actions">
                                    <a href="${item.link}" class="btn-view" target="_blank">View Product</a>
                                    <button class="remove-item-page"><i class="fa fa-trash"></i></button>
                                 </div>
                            </div>
                        </div>
                    </li>`;
            });
            $('#cart-items-page').html(pageHTML);
        }

        // --- Update Add-to-Cart Button State ---
        $('.add-to-custom-cart').each(function () {
            const $btn = $(this);
            const pid = $btn.data('product-id');
            const $wrap = $btn.closest('.custom-cart-wrap');

            if (data.items.find(i => i.id == pid)) {
                // Product already in cart → show tick, hide qty input
                $btn.addClass('added').html('<i class="fa fa-check added-icon" aria-hidden="true"></i>');
                $wrap.find('.custom-cart-qty').hide();
            } else {
                // Product not in cart → reset button & show qty input
                $btn.removeClass('added').html('<i class="fa fa-cart-plus" aria-hidden="true"></i>');
                $wrap.find('.custom-cart-qty').show();
            }
        });
    }


    // ================= Notification Box =================
    function notify(msg) {
        if (!msg) return;
        $('.cart-success').remove(); // Remove existing notifications
        const box = $(`<div class="cart-success">${msg}</div>`);
        $('body').append(box);
        setTimeout(() => box.fadeOut(400, () => box.remove()), 2200); // Auto-hide after 2.2s
    }


    // ================= Initial Sync on Page Load =================
    // Ensures buttons reflect server cart state
    $.post(customCartAjax.ajaxurl, {
        action: 'get_custom_cart',
        nonce: customCartAjax.nonce
    }, function (res) {
        if (res && res.success) {
            refreshUI(res.data);
        }
    });

});
