<?php
/*
Plugin Name: Custom WooCommerce PDF Cart
Description: Custom WooCommerce cart with quantity, notifications, PDF quote generation, phone field, thumbnails, and thank-you screen.
Version: 4.5
Author: Sangam Kumar Maurya
Author Email: sangamkmaurya93@gmail.com
*/

if (!defined('ABSPATH')) exit; // Prevent direct access


/** ---------------- Enqueue Plugin CSS & JS ---------------- */
add_action('wp_enqueue_scripts', function () {
    // Load plugin stylesheet
    wp_enqueue_style(
        'custom-cart-style', 
        plugin_dir_url(__FILE__) . 'style.css', 
        [], 
        '4.5'
    );

    // Load Font Awesome CDN for icons
    wp_enqueue_style(
        'font-awesome', 
        'https://maxcdn.bootstrapcdn.com/font-awesome/4.5.0/css/font-awesome.min.css', 
        [], 
        '4.5'
    );

    // Load plugin JS file
    wp_enqueue_script(
        'custom-cart-js', 
        plugin_dir_url(__FILE__) . 'script.js', 
        ['jquery'], 
        '4.5', 
        true
    );

    // Pass dynamic data to JS for AJAX operations
    $form_page_url = home_url('/custom-cart-form');
    wp_localize_script('custom-cart-js', 'customCartAjax', [
        'ajaxurl'       => admin_url('admin-ajax.php'),
        'nonce'         => wp_create_nonce('custom_cart_nonce'),
        'home_url'      => home_url('/'),
        'form_page_url' => $form_page_url,
    ]);
});


/** ---------------- Start PHP Session for Custom Cart ---------------- */
add_action('init', function () {
    if (!session_id()) session_start(); // Start session if not already started

    // Initialize cart array in session if not set
    if (!isset($_SESSION['custom_cart']) || !is_array($_SESSION['custom_cart'])) {
        $_SESSION['custom_cart'] = [];
    }
}, 1);


/** ---------------- Helper Functions ---------------- */

/**
 * Safely get WooCommerce product price and object
 * @param int $product_id
 * @return array [price, WC_Product object|null]
 */
function pw_custom_cart_safe_price($product_id) {
    $product = wc_get_product($product_id);
    if (!$product) return [0, null]; // Return 0 if product not found
    return [(float) $product->get_price(), $product];
}

/**
 * Get structured cart data (for AJAX responses)
 * @param string $msg Optional status message
 * @return array Cart details
 */
function pw_custom_cart_get_data($msg = '') {
    $items = [];
    $total = 0;

    foreach ((array) $_SESSION['custom_cart'] as $cid => $row) {
        [$price, $product] = pw_custom_cart_safe_price($cid);
        if (!$product) continue;

        $qty = max(1, intval($row['qty'])); // Ensure quantity >= 1
        $subtotal = $price * $qty;
        $total += $subtotal;
        $stock_qty = $product->get_stock_quantity() ?: 9999;

        $items[] = [
            'id'    => $cid,
            'title' => get_the_title($cid),
            'price' => html_entity_decode(wp_strip_all_tags(wc_price($subtotal))),
            'link'  => get_permalink($cid),
            'img'   => get_the_post_thumbnail_url($cid, 'thumbnail'),
            'qty'   => $qty,
            'stock' => $stock_qty,
        ];
    }

    return [
        'count'   => count($_SESSION['custom_cart']),
        'items'   => $items,
        'total'   => html_entity_decode(wp_strip_all_tags(wc_price($total))),
        'message' => $msg,
    ];
}


/** ---------------- AJAX: Get Current Cart Data ---------------- */
add_action('wp_ajax_get_custom_cart', 'pw_ajax_get_custom_cart');
add_action('wp_ajax_nopriv_get_custom_cart', 'pw_ajax_get_custom_cart');
function pw_ajax_get_custom_cart() {
    check_ajax_referer('custom_cart_nonce', 'nonce'); // Security check
    wp_send_json_success(pw_custom_cart_get_data()); // Return current cart data
}


/** ---------------- Single Product Page: Quantity Input + Custom Add-to-Cart ---------------- */
add_action('woocommerce_after_add_to_cart_button', function () {
    global $product;
    if (!$product) return;

    // Output custom quantity input + add-to-cart button
    echo '<div class="custom-cart-wrap">';
    echo '<input type="number" class="custom-cart-qty" value="1" min="1" max="' . esc_attr($product->get_stock_quantity() ?: 9999) . '">';
    echo '<button type="button" class="add-to-custom-cart" data-product-id="' . esc_attr($product->get_id()) . '">
            <i class="fa fa-cart-plus" aria-hidden="true"></i>
          </button>';
    echo '</div>';
});


/** ---------------- Product Loop (Shop / Category Pages) ---------------- */
// Remove default title
remove_action('woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_title', 10);

add_action('woocommerce_after_shop_loop_item', function () {
    global $product;
    if (!$product) return;

    echo '<div class="custom-loop-product" style="text-align:center;">';

    // Product Title
    echo '<div class="custom-loop-title" style="font-weight:normal; margin:10px 0;">' . get_the_title($product->get_id()) . '</div>';

    // Quantity + Add to Cart Button
    echo '<div class="custom-cart-wrap" style="margin-top:5px;">';
    echo '<input type="number" class="custom-cart-qty" value="1" min="1" max="' . esc_attr($product->get_stock_quantity() ?: 9999) . '" style="width:50px; text-align:center; margin-right:5px;">';
    echo '<button class="add-to-custom-cart" data-product-id="' . esc_attr($product->get_id()) . '" style="padding:5px 10px;">
            <i class="fa fa-cart-plus" aria-hidden="true"></i>
          </button>';
    echo '</div>';

    echo '</div>';
}, 20);


/** ---------------- Footer Mini-Cart / Popup ---------------- */
add_action('wp_footer', function () {
    $cart  = (array) $_SESSION['custom_cart'];
    $count = count($cart);
    $total = 0;

    // Mini-cart button (shows total count)
    echo '<div id="custom-cart-button"><i class="fa fa-cart-plus"></i> (<span id="cart-count">' . intval($count) . '</span>)</div>';

    // Mini-cart popup HTML
    echo '<div id="custom-cart-popup" class="modern-popup">
            <h4>Your Selection</h4>
            <ul class="cart-items">';

    foreach ($cart as $id => $row) {
        [$price, $product] = pw_custom_cart_safe_price($id);
        if (!$product) continue;

        $qty = max(1, intval($row['qty']));
        $subtotal = $price * $qty;
        $total += $subtotal;
        $stock_qty = $product->get_stock_quantity() ?: 9999;

        echo '<li data-id="' . esc_attr($id) . '" data-max-qty="'. intval($stock_qty) .'">
                <img src="' . esc_url(get_the_post_thumbnail_url($id, 'thumbnail')) . '" class="cart-thumb"/>
                <div class="cart-info">
                    <strong>' . esc_html(get_the_title($id)) . '</strong><br>
                    Qty: <span class="item-qty">' . $qty . '</span>
                    <button class="qty-minus">➖</button>
                    <button class="qty-plus">➕</button><br>
                    <span class="cart-price">' . wc_price($subtotal) . '</span>
                </div>
                <button class="remove-item"><i class="fa fa-trash"></i></button>
                </li>';
    }

    echo '</ul>
          <div class="cart-total">Total: <span id="cart-total">' . wc_price($total) . '</span></div>
          <button id="proceed-to-form" class="modern-proceed">Request Quote</button>
          </div>';
});


/** ---------------- AJAX: Add / Update / Remove Cart Items ---------------- */
// Add product to custom cart
add_action('wp_ajax_add_to_custom_cart', 'pw_ajax_add_to_custom_cart');
add_action('wp_ajax_nopriv_add_to_custom_cart', 'pw_ajax_add_to_custom_cart');
function pw_ajax_add_to_custom_cart() {
    check_ajax_referer('custom_cart_nonce', 'nonce');

    $id  = intval($_POST['product_id'] ?? 0);
    $qty = max(1, intval($_POST['quantity'] ?? 1));
    $product = wc_get_product($id);

    if (!$id || !$product) wp_send_json_error(['message' => 'Invalid product']);

    // Limit quantity to available stock
    $max_qty = $product->get_stock_quantity() ?: 9999;
    if ($qty > $max_qty) $qty = $max_qty;

    $_SESSION['custom_cart'][$id] = ['qty' => $qty];

    wp_send_json_success(pw_custom_cart_get_data('✅ Product added to cart!'));
}

// Update quantity in cart
add_action('wp_ajax_update_cart_quantity', 'pw_ajax_update_cart_quantity');
add_action('wp_ajax_nopriv_update_cart_quantity', 'pw_ajax_update_cart_quantity');
function pw_ajax_update_cart_quantity() {
    check_ajax_referer('custom_cart_nonce', 'nonce');

    $id  = intval($_POST['product_id'] ?? 0);
    $qty = max(1, intval($_POST['quantity'] ?? 1));

    if (!$id || !isset($_SESSION['custom_cart'][$id])) wp_send_json_error(['message' => 'Invalid product']);

    $product = wc_get_product($id);
    $max_qty = $product->get_stock_quantity() ?: 9999;

    if ($qty < 1) wp_send_json_error(['message' => '❌ Quantity cannot be less than 1']);
    if ($qty > $max_qty) wp_send_json_error(['message' => '❌ More than available quantity not available']);

    $_SESSION['custom_cart'][$id]['qty'] = $qty;
    wp_send_json_success(pw_custom_cart_get_data('🔄 Quantity updated!'));
}

// Remove product from cart
add_action('wp_ajax_remove_from_custom_cart', 'pw_ajax_remove_from_custom_cart');
add_action('wp_ajax_nopriv_remove_from_custom_cart', 'pw_ajax_remove_from_custom_cart');
function pw_ajax_remove_from_custom_cart() {
    check_ajax_referer('custom_cart_nonce', 'nonce');
    $id = intval($_POST['product_id'] ?? 0);
    unset($_SESSION['custom_cart'][$id]);
    wp_send_json_success(pw_custom_cart_get_data('🗑️ Product removed!'));
}


/** ---------------- Shortcode: Custom Cart Form + Thank You Page ---------------- */
add_shortcode('custom_cart_form', function () {

    // ---------------- Thank You Page ----------------
    if (isset($_GET['submitted']) && $_GET['submitted'] === 'true') {
        ob_start(); ?>
        <div class="thankyou-wrap">
            <h1 class="thankyou-heading">Thank You!</h1>
            <p class="thankyou-sub">Your quote request has been submitted successfully.<br>Please check your mail.</p>
            <a class="go-home-btn" href="<?php echo esc_url(home_url('/')); ?>">Go to Home</a>
        </div>
        <?php
        return ob_get_clean();
    }

    // ---------------- Cart Form ----------------
    $cart = (array) $_SESSION['custom_cart'];
    $total = 0;
    ob_start(); ?>
    <div class="custom-form-wrap">

        <!-- Cart Summary -->
        <div class="cart-summary">
            <h3>Your Selected Products (<span id="cart-count-page"><?php echo count($cart); ?></span>)</h3>
            <div class="cart-list-scroll">
                <ul id="cart-items-page">
                    <?php foreach ($cart as $id => $row):
                        [$price, $product] = pw_custom_cart_safe_price($id); 
                        if(!$product) continue;
                        $qty = max(1, intval($row['qty'])); 
                        $subtotal = $price * $qty; 
                        $total += $subtotal;
                        $stock_qty = $product->get_stock_quantity() ?: 9999;
                        ?>
                        <li data-id="<?php echo esc_attr($id); ?>" data-max-qty="<?php echo intval($stock_qty); ?>">
                            <div class="cart-item-box">
                                <img src="<?php echo esc_url(get_the_post_thumbnail_url($id, 'thumbnail')); ?>" class="cart-item-thumb" alt="">
                                <div class="cart-item-content">
                                    <div class="cart-item-title"><?php echo esc_html(get_the_title($id)); ?></div>
                                    <div class="cart-item-details">
                                        Qty: <span class="item-qty"><?php echo intval($qty); ?></span>
                                        <button class="qty-minus">➖</button>
                                        <button class="qty-plus">➕</button>
                                        - <?php echo html_entity_decode(wp_strip_all_tags(wc_price($subtotal))); ?>
                                    </div>
                                    <div class="cart-item-actions">
                                        <a class="btn-view" href="<?php echo esc_url(get_permalink($id)); ?>" target="_blank">View Product</a>
                                        <button class="remove-item-page"><i class="fa fa-trash"></i></button>
                                    </div>
                                </div>
                            </div>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <div class="cart-total">Total: <span id="cart-total-page"><?php echo html_entity_decode(wp_strip_all_tags(wc_price($total))); ?></span></div>
        </div>

        <!-- Customer Form -->
        <div class="cart-form">
            <form method="post" id="custom-cart-form">
                <?php wp_nonce_field('custom_cart_submit','custom_cart_submit_nonce'); ?>

                <label>Full Name:
                    <input type="text" name="cust_name" required placeholder="Enter full name">
                </label>

                <label>Email Address:
                    <input type="email" name="cust_email" required placeholder="Enter email address">
                </label>

                <label>Phone Number:
                    <input type="text" name="cust_phone" required placeholder="Enter 10-digit phone"
                           pattern="\d{10}" maxlength="10"
                           title="Please enter a valid 10-digit phone number">
                </label>

                <button type="submit" name="custom_cart_submit" class="submit-btn">Mail Quote</button>
            </form>
        </div>
    </div>
    <?php
    return ob_get_clean();
});


/** ---------------- PDF Generation, Email, and Redirect ---------------- */
add_action('init', function () {
    // Exit if form not submitted
    if (!isset($_POST['custom_cart_submit'])) return;
    if (!isset($_POST['custom_cart_submit_nonce']) || !wp_verify_nonce($_POST['custom_cart_submit_nonce'], 'custom_cart_submit')) return;

    // Sanitize input
    $name  = sanitize_text_field($_POST['cust_name'] ?? '');
    $email = sanitize_email($_POST['cust_email'] ?? '');
    $phone = sanitize_text_field($_POST['cust_phone'] ?? '');
    if (!is_email($email)) return;

    $cart = (array) $_SESSION['custom_cart'];
    if (empty($cart)) return;

    // Load TCPDF library
    require_once plugin_dir_path(__FILE__) . 'pdf-lib/tcpdf.php';
    $pdf = new TCPDF(); 
    $pdf->AddPage(); 
    $pdf->SetFont('dejavusans','',12);

    // PDF Header with company & customer info
    $header_html = '
        <h1 style="text-align:center; font-size:20pt; margin-top:30px; margin-bottom:10px; color:#333;">My Company Pvt Ltd</h1>
        <p style="text-align:center; font-size:11pt; margin:0 0 20px 0; color:#555;">
        123 Business Street, Mumbai, India | +91-00000 00000 | info@company.com
        </p>
        <hr style="border:0; border-top:1px solid #000; margin:0 0 25px 0;">
        <div style="font-size:12pt; line-height:1.5; margin-bottom:30px;">
            <p style="margin:0;"><b>Name:</b> ' . esc_html($name) . '</p>
            <p style="margin:0;"><b>Email:</b> ' . esc_html($email) . '</p>
            <p style="margin:0;"><b>Phone:</b> ' . esc_html($phone) . '</p>
        </div>';
    $pdf->writeHTML($header_html, true, false, true, false, '');

    // Cart Table in PDF
    $tbl = '<table border="1" cellpadding="6">
                <thead>
                    <tr>
                        <th><b>Image</b></th>
                        <th><b>Product</b></th>
                        <th><b>Unit Price</b></th>
                        <th><b>Quantity</b></th>
                        <th><b>Link</b></th>
                    </tr>
                </thead>
                <tbody>';
    $total = 0;
    foreach ($cart as $id=>$row) {
        [$price,$product] = pw_custom_cart_safe_price($id);
        if(!$product) continue;
        $qty = max(1, intval($row['qty']));
        $subtotal = $price * $qty;
        $total += $subtotal;

        $thumb_url = get_the_post_thumbnail_url($id, 'thumbnail');
        $thumb_html = $thumb_url ? '<img src="'.$thumb_url.'" width="40" height="40">' : '';

        $tbl .= '<tr>
                    <td align="center">'.$thumb_html.'</td>
                    <td>'. esc_html(get_the_title($id)) .'</td>
                    <td>'. html_entity_decode(wp_strip_all_tags(wc_price($price))) .'</td>
                    <td>'. $qty .'</td>
                    <td><a href="'. esc_url(get_permalink($id)) .'">View</a></td>
                 </tr>';
    }
    $tbl .= '<tr>
                <td colspan="4" align="right"><b>Total</b></td>
                <td><b>'. html_entity_decode(wp_strip_all_tags(wc_price($total))) .'</b></td>
             </tr>';
    $tbl .= '</tbody></table>';
    $pdf->writeHTML($tbl, true, false, false, false, '');

    // Save PDF in uploads folder
    $upload_dir = wp_upload_dir();
    $sanitized_name = preg_replace('/[^A-Za-z0-9_\-]/', '_', $name);
    $filename       = 'quote_' . $sanitized_name . '_' . time() . '.pdf';
    $pdf_path       = trailingslashit($upload_dir['basedir']) . $filename;

    $pdf->Output($pdf_path, 'F');

    // Send emails to customer & admin
    $headers = ['Content-Type: text/html; charset=UTF-8'];
    $body    = 'Hello '. esc_html($name) .',<br>Your quote PDF is attached.<br><b>Phone:</b> '. esc_html($phone) .'<br><b>Email:</b> '. esc_html($email);

    wp_mail($email, 'Your Quote', $body, $headers, [$pdf_path]);
    wp_mail(get_option('admin_email'), 'New Cart Submission', $body, $headers, [$pdf_path]);

    // Clear cart session after submission
    unset($_SESSION['custom_cart']);

    // Redirect to Thank You page
    $redirect = !empty($_SERVER['HTTP_REFERER']) ? esc_url_raw($_SERVER['HTTP_REFERER']) : home_url('/custom-cart-form');
    $redirect = add_query_arg('submitted', 'true', $redirect);
    wp_safe_redirect($redirect);
    exit;
});
